﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace DocumentProcessing
{
    class DocumentProcessor
    {
        // Listing 5-25. Zaktualizowane składowe zdarzeń
        public event EventHandler<ProcessEventArgs> Processing;
        public event EventHandler<ProcessEventArgs> Processed;

        public Func<Document, string> LogTextProvider
        {
            get;
            set;
        }


        class ActionCheckPair
        {
            public Action<Document> Action { get; set; }
            public Predicate<Document> QuickCheck { get; set; }
        }

        private readonly List<ActionCheckPair> processes =
            new List<ActionCheckPair>();

        public void AddProcess(Action<Document> action)
        {
            AddProcess(action, null);
        }

        public void AddProcess(Action<Document> action,
            Predicate<Document> quickCheck)
        {
            processes.Add(
            new ActionCheckPair { Action = action, QuickCheck = quickCheck });
        }

        // Listing 5-27. Tworzenie obiektu argumentu zdarzenia
        public void Process(Document doc)
        {
            ProcessEventArgs e = new ProcessEventArgs(doc);
            OnProcessing(e);
            // Najpierw przeprowadzamy szybką weryfikację dokumentu.
            foreach (ActionCheckPair process in processes)
            {
                if (process.QuickCheck != null && !process.QuickCheck(doc))
                {
                    Console.WriteLine("Przetwarzanie nie zakończy się pomyślnie.");
                    if (LogTextProvider != null)
                    {
                        Console.WriteLine(LogTextProvider(doc));
                    }
                    OnProcessed(e);
                    return;
                }
            }
            // Teraz wykonujemy akcję.
            foreach (ActionCheckPair process in processes)
            {
                process.Action(doc);
                if (LogTextProvider != null)
                {
                    Console.WriteLine(LogTextProvider(doc));
                }
            }
            OnProcessed(e);
        }

        // Listing 5-26. Zaktualizowany kod generujący zdarzenia
        private void OnProcessing(ProcessEventArgs e)
        {
            if (Processing != null)
            {
                Processing(this, e);
            }
        }

        private void OnProcessed(ProcessEventArgs e)
        {
            if (Processed != null)
            {
                Processed(this, e);
            }
        }
    }
}